#!/bin/bash

set -e

###
# 
# This file is part of yaourtix, and meant to be user-editable.
# 
# It is used when the source (i.e. PKGBUILD and accompanying files) of
# some binary package should be downloaded (e.g. with the '-G'-option or
# when a customizepkg-hook should be applied) to determine how to
# download the source.
# 
# This file is a bash script, and it takes the following arguments:
# 
#   $1: repo: The name of the repository the binary package resides in.
#   $2: pkgbase: The name of the package/ pkgbase of the package.
#   $3: arch: The architecture the package is for.
# 
# This script needs to output the follwing to stdout, separated by
# newline:
# 
#   fetcher: A complete command (including URL and extraction commands)
#            to retrieve the package sources, unpacked and ready to use.
# 
# Since this file is a bash-script executed by yaourtix, be careful when
# editing it.
# 
###

repo="$1"
pkgbase="$2"
arch="$3"

fetcher='/bin/false' # If this does not get set to something meaningful below, bail out when trying to execute it.

msg() {
  # cat <<< "$@"
  printf '%s\n' "$@"
}

errmsg() {
  msg "$@" > /dev/stderr
}

case "${repo}" in

  ### Arch Linux:
  'core'|'extra'|'testing') # Arch Linux basic repositories, including testing
    _url="https://git.archlinux.org/svntogit/packages.git/snapshot/packages/${pkgbase}.tar.gz"
    fetcher="curl -L \"${_url}\" -o - | bsdtar --strip-components 3 --include=\"*/${pkgbase}/trunk/\" -xvf -"
  ;;
  'community'|'multilib') # Arch Linux community repository
    _url="https://git.archlinux.org/svntogit/community.git/snapshot/community-packages/${pkgbase}.tar.gz"
    fetcher="curl -L \"${_url}\" -o - | bsdtar --strip-components 3 --include=\"*/${pkgbase}/trunk/\" -xvf -"
  ;;
#   'staging') # Arch Linux staging repository
#     # Please fill with correct code.
#   ;;
#   'community-staging') # Arch Linux community staging repository
#     # Please fill with correct code.
#   ;;
#   'community-testing') # Arch Linux community testing repository
#     # Please fill with correct code.
#   ;;
#   'multilib-staging') # Arch Linux multilib staging repository
#     # Please fill with correct code.
#   ;;
#   'multilib-testing') # Arch Linux multilib testing repository
#     # Please fill with correct code.
#   ;;
  
  ### Artix Linux:
  'system'|'world'|'galaxy'|'lib32'|'gremlins'|'goblins'|'galaxy-gremlins'|'lib32-gremlins'|'galaxy-goblins'|'lib32-goblins') # Artix Linux repositories, including Artix' testing (= 'gremlins') and Artix' staging (= 'goblins').
    case "${repo}" in
      'system')
        _archrepo='core'
      ;;
      'world')
        _archrepo='extra'
      ;;
      'galaxy')
        _archrepo='community'
      ;;
      'lib32')
        _archrepo='multilib'
      ;;
      'gremlins')
        _archrepo='testing'
      ;;
      'goblins')
        _archrepo='staging'
      ;;
      'galaxy-gremlins')
        _archrepo='community-testing'
      ;;
      'lib32-gremlins')
        _archrepo='multilib-testing'
      ;;
      'galaxy-goblins')
        _archrepo='community-staging'
      ;;
      'lib32-goblins')
        _archrepo='multilib-staging'
      ;;
      *)
        errmsg "repo '${repo}' not defined in Artix Linux repository name check."
        errmsg "This is an internal bug of '${BASH_SOURCE[0]}'."
        errmsg "Aborting."
        exit 1
      ;;
    esac
    _suffix="$(echo ${pkgbase:0:1} | tr '[:lower:]' '[:upper:]')"
    fetcher="git clone \"https://gitea.artixlinux.org/packages${_suffix}/${pkgbase}.git\" \"${pkgbase}\" && mv \"${pkgbase}/repos/${_archrepo}-${arch}\"/* . && rm -Rf \"${pkgbase}\""
  ;;
  
  ### Other (add your repositories here -- one example is given):
  'archstrike'|'archstrike-testing') # Unofficial user repository 'archstrike'
    fetcher="svn export https://github.com/ArchStrike/ArchStrike.git/trunk/\"${repo}\"/\"${pkgbase}\" \"${pkgbase}\" && mv \"${pkgbase}\"/* . && rm -Rf \"${pkgbase}\""
  ;;
  
  *) # Bail out when package sources from a repository not handled here should be fetched.
    errmsg "Trying to fetch sources for '${repo}/${pkgbase}'."
    errmsg "However, there is no rule defined how to fetch sources from repository '${repo}'."
    errmsg ""
    errmsg "Please edit '${BASH_SOURCE[0]}' to add such a rule."
    errmsg ""
    errmsg "Not fetching, aborting."
    exit 1
  ;;
esac

cat << EOO
${fetcher}
EOO

exit "$?"
